// listview.cpp : implementation file
//

#include "stdafx.h"
#include "uvikon.h"
#include "listview.h"
#include "titleedi.h"

#include <afxwin.h>

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif


// statics for floating menu
static char szFMUndo[] = "Undo";
static char szFMCut[] = "Cut";
static char szFMCopy[] = "Copy";
static char szFMPaste[] = "Paste";
static char szFMDelete[] = "Delete";
static char szFMDeleteAll[] = "Delete all";

/////////////////////////////////////////////////////////////////////////////
// CListView

IMPLEMENT_DYNCREATE(CListView, CScrollView)

BEGIN_MESSAGE_MAP(CListView, CScrollView)
	//{{AFX_MSG_MAP(CListView)
	ON_COMMAND(ID_EDIT_CLEAR, OnEditClear)
	ON_UPDATE_COMMAND_UI(ID_EDIT_CLEAR, OnUpdateEditClear)
	ON_COMMAND(ID_EDIT_CLEAR_ALL, OnEditClearAll)
	ON_UPDATE_COMMAND_UI(ID_EDIT_CLEAR_ALL, OnUpdateEditClearAll)
	ON_COMMAND(ID_EDIT_COPY, OnEditCopy)
	ON_UPDATE_COMMAND_UI(ID_EDIT_COPY, OnUpdateEditCopy)
	ON_COMMAND(ID_EDIT_CUT, OnEditCut)
	ON_UPDATE_COMMAND_UI(ID_EDIT_CUT, OnUpdateEditCut)
	ON_COMMAND(ID_EDIT_PASTE, OnEditPaste)
	ON_UPDATE_COMMAND_UI(ID_EDIT_PASTE, OnUpdateEditPaste)
	ON_COMMAND(ID_EDIT_UNDO, OnEditUndo)
	ON_UPDATE_COMMAND_UI(ID_EDIT_UNDO, OnUpdateEditUndo)
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_MOUSEMOVE()
	ON_WM_KILLFOCUS()
	ON_WM_LBUTTONDBLCLK()
	ON_WM_RBUTTONDOWN()
	ON_WM_SETFOCUS()
	ON_WM_KEYDOWN()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

CListView::CListView()
{
	m_nPrevLine = -1;
}

CListView::~CListView()
{
}

/////////////////////////////////////////////////////////////////////////////
// CUvikonView drawing

/////////////////////////////////////////////////////////////////////////////
// OnDraw(): called when redrawing of data is necessary
// All DOC_XXX layout constants are defined in uvikon.h
////////////////////////////////////////////////////////////////////////////
void CListView::OnDraw(CDC* pDC)    
{
	int i, nSelState, prevROP2, nXExtent;
	CPoint pointArray[5];
	CFont* pOldFont;
	CPen *pOldPen, *pOldPen1;
	CBrush *pOldBrush;
	CString* pszTitle;
	CSize actualSize, maxSize;
	BOOL bSelect = FALSE;
	COLORREF nTextNormal;
	COLORREF nTextSelect;
	COLORREF nBkNormal;
	COLORREF nBkSelect;
	CRect rect, rectClip, rectItem;
	
	// get system colors for a well-behaved windows app
	nTextSelect = GetSysColor(COLOR_HIGHLIGHTTEXT);
	nBkSelect = GetSysColor(COLOR_HIGHLIGHT);
	nTextNormal = GetSysColor(COLOR_WINDOWTEXT);
	nBkNormal = GetSysColor(COLOR_WINDOW);
	
	// initialize pen and brush for drawing rectangles
	CPen penDash(PS_DASH, 1, nBkSelect);
	CPen penSolid(PS_SOLID, 1, nBkSelect);
	CBrush brush(nBkSelect);
	pOldPen = pDC->SelectObject(&penSolid);
	pOldBrush = pDC->SelectObject(&brush);
	
	CUvikonDoc* pDoc = (CUvikonDoc*)GetDocument();
	ASSERT_VALID(pDoc);
    
	maxSize = pDoc->GetDocSize();   // get old size of document
	pOldFont = (CFont*)pDC->SelectStockObject(/*ANSI_FIXED_FONT`*/SYSTEM_FONT);  // select font
	
	// compute x-dimension of hilight bar
	GetClientRect(&rect);
	nXExtent = (rect.right > maxSize.cx) ? rect.right : maxSize.cx;
	
	// get invalidated clipping rect
	pDC->GetClipBox(&rectClip);
	
	for (i = 0; i < pDoc->GetNumRun(); i++)    // run thru all runs
	{
		// bounding rectangle of current run
		rectItem.left = 0;
		rectItem.right = nXExtent;
		rectItem.top = i*(m_tm.tmHeight + DOC_LINESPACE);
		rectItem.bottom = (i+1)*(m_tm.tmHeight + DOC_LINESPACE);
		
		if (!rectItem.IntersectRect(&rectItem, &rectClip))
		{
			continue;    // skip this line
		}
		
		nSelState = pDoc->GetIsSelected(i);
		
		// to speed up the drawing, we do not set text and background color for each
		// line individually. Instead, we keep the select state of the previous line 
		// in bSelect. Only if the selection state changes from the previous to this
		// line, we will call the GDI functions.
		if (nSelState > 0 && !bSelect)
		{
			pDC->SetTextColor(nTextSelect);
			pDC->SetBkColor(nBkSelect);
			bSelect = TRUE;	
		}
		else if (nSelState == 0 && bSelect)
		{
			pDC->SetTextColor(nTextNormal);
			pDC->SetBkColor(nBkNormal);
			bSelect = FALSE;
		}
		
		// if this line is selected, we first draw a rect
		if (nSelState > 0)
		{
			pDC->Rectangle(DOC_LEFTMARGIN + DOC_ICONWIDTH + DOC_ICONSPACE, 
						   i*(m_tm.tmHeight + DOC_LINESPACE), 
						   nXExtent, 
						   (i+1)*(m_tm.tmHeight + DOC_LINESPACE));
			
			// if this line is the ONE current line, we draw a line indicating the
			// input focus (if this wnd has the input focus)
			if (nSelState == SEL_CURRENT && GetFocus() == this)
			{
				pointArray[0] = CPoint(DOC_LEFTMARGIN + DOC_ICONWIDTH + DOC_ICONSPACE, 
									   i*(m_tm.tmHeight + DOC_LINESPACE));
				pointArray[1] = CPoint(nXExtent - 1, i*(m_tm.tmHeight + DOC_LINESPACE));
				pointArray[2] = CPoint(nXExtent - 1, (i+1)*(m_tm.tmHeight + DOC_LINESPACE) - 1);
				pointArray[3] = CPoint(DOC_LEFTMARGIN + DOC_ICONWIDTH + DOC_ICONSPACE, 
									   (i+1)*(m_tm.tmHeight + DOC_LINESPACE) - 1);
				pointArray[4] = pointArray[0];
				
				prevROP2 = pDC->SetROP2(R2_NOT);
				pOldPen1 = pDC->SelectObject(&penDash);
				pDC->Polyline(pointArray, 5);
				pDC->SetROP2(prevROP2);
				pDC->SelectObject(pOldPen1);
			}
		}
		
		// get the title string
		pszTitle = pDoc->GetRunTitle(i);
		if (pszTitle == NULL)
		{
			continue;   // unexpected error
		}
		
		// draw the icon  
		if (pszTitle->Find("SCAN") != -1)
		{
			DrawIcon(MET_LSCAN, pDC, i);
		}
		else if (pszTitle->Find("LFIX") != -1)
		{
			DrawIcon(MET_LFIX, pDC, i);
		}
		else if (pszTitle->Find("TDRV") != -1)
		{
			DrawIcon(MET_TDRIVE, pDC, i);
		}
		
		// now we actually print the title, discarding the method identifier
		pDC->TabbedTextOut(2*DOC_LEFTMARGIN + DOC_ICONWIDTH + DOC_ICONSPACE, 
						   i*(m_tm.tmHeight + DOC_LINESPACE)+1, 
						   pszTitle->Right(pszTitle->GetLength() - 5), 
						   pszTitle->GetLength() - 5, 
						   0, 
						   NULL, 
						   0);
	}
	
	// restore old GDI objects
	pDC->SelectObject(pOldFont);
	pDC->SelectObject(pOldPen);
	pDC->SelectObject(pOldBrush);
}

////////////////////////////////////////////////////////////////////////////////
// DrawIcon(): Draws the method icon at the start of each line
// Arguments: nType: Denotes the type of icon to draw
//            pDC: device context
//            nLine: current line
////////////////////////////////////////////////////////////////////////////////
void CListView::DrawIcon(int nType, CDC* pDC, int nLine)
{
	CBrush *pOldBrush;
	CPen *pOldPen;
    CPoint upperleft;
    
	CPen penGray(PS_SOLID, 1, RGB(190, 190, 190));
	CPen penDkGray(PS_SOLID, 1, RGB(100, 100, 100));
	CPen penGreenWide(PS_SOLID, 2, RGB(0, 255, 0));
	CPen penCyan(PS_SOLID, 1, RGB(0, 255, 255));
	CPen penRed(PS_SOLID, 1, RGB(255, 0, 0));
	
	pOldBrush = (CBrush*)pDC->SelectStockObject(LTGRAY_BRUSH);
	pOldPen = (CPen*)pDC->SelectStockObject(BLACK_PEN);
		
	upperleft.x = DOC_LEFTMARGIN;
	upperleft.y = nLine*(m_tm.tmHeight + DOC_LINESPACE) + 1;
	
	// draw background
	pDC->Rectangle(DOC_LEFTMARGIN,
				   upperleft.y, 
				   DOC_LEFTMARGIN + DOC_ICONWIDTH, 
				   upperleft.y + DOC_ICONHEIGHT);
	
	// make 3d: white line left+top
	pDC->SelectStockObject(WHITE_PEN);
	pDC->MoveTo(DOC_LEFTMARGIN + DOC_ICONWIDTH - 2, upperleft.y + 1);
	pDC->LineTo(DOC_LEFTMARGIN + 1, upperleft.y + 1);
	pDC->LineTo(DOC_LEFTMARGIN + 1, upperleft.y + DOC_ICONHEIGHT - 2);				   
	
	// make 3d: dark gray line bottom+right
	pDC->SelectObject(&penDkGray);
	pDC->LineTo(DOC_LEFTMARGIN + DOC_ICONWIDTH - 2, upperleft.y + DOC_ICONHEIGHT - 2);
	pDC->LineTo(DOC_LEFTMARGIN + DOC_ICONWIDTH - 2, upperleft.y + 1);
	
	// draw lambda
	switch(nType)
	{
		case MET_LSCAN:       // both need a lambda
		case MET_LFIX:
			pDC->SelectStockObject(BLACK_PEN);
			pDC->MoveTo(upperleft.x + 5, upperleft.y + 2);
			pDC->LineTo(upperleft.x + 11, upperleft.y + 8);
			pDC->MoveTo(upperleft.x + 6, upperleft.y + 2);
			pDC->LineTo(upperleft.x + 12, upperleft.y + 8);
			pDC->MoveTo(upperleft.x + 7, upperleft.y + 5);
			pDC->LineTo(upperleft.x + 3, upperleft.y + 8);
			pDC->SetPixel(upperleft.x + 4, upperleft.y + 2, RGB(0, 0, 0));
			break;
		case MET_TDRIVE:
			break;
	}
	
	// draw rest
	switch(nType)
	{
		case MET_LSCAN:
			// left part of arrow in cyan
			pDC->SelectObject(&penCyan);
			pDC->MoveTo(upperleft.x + 3, upperleft.y + 11);
			pDC->LineTo(upperleft.x + 8, upperleft.y + 11);
			pDC->MoveTo(upperleft.x + 4, upperleft.y + 10);
			pDC->LineTo(upperleft.x + 4, upperleft.y + 13);
			pDC->MoveTo(upperleft.x + 5, upperleft.y + 9);
			pDC->LineTo(upperleft.x + 5, upperleft.y + 14);
            
            // right part of arrow in red
			pDC->SelectObject(&penRed);
			pDC->MoveTo(upperleft.x + 8, upperleft.y + 11);
			pDC->LineTo(upperleft.x + 13, upperleft.y + 11);
			pDC->MoveTo(upperleft.x + 12, upperleft.y + 10);
			pDC->LineTo(upperleft.x + 12, upperleft.y + 13);
			pDC->MoveTo(upperleft.x + 11, upperleft.y + 9);
			pDC->LineTo(upperleft.x + 11, upperleft.y + 14);
			break;
			
		case MET_LFIX:
			// simple vertical line in green
			pDC->SelectObject(&penGreenWide);
			pDC->MoveTo(upperleft.x + 7, upperleft.y + 8);
			pDC->LineTo(upperleft.x + 7, upperleft.y + 12);
			break;
			
		case MET_TDRIVE:
			// clock face in black on white background
			pDC->SelectStockObject(WHITE_BRUSH);
			pDC->Ellipse(upperleft.x + 2, upperleft.y + 2, 
						 upperleft.x + DOC_ICONWIDTH - 2, upperleft.y + DOC_ICONHEIGHT - 2);
			
			// the tick marks
			pDC->SetPixel(upperleft.x + 4, upperleft.y + 9, RGB(0, 0, 0));
			pDC->SetPixel(upperleft.x + DOC_ICONWIDTH - 4, upperleft.y + 9, RGB(0, 0, 0));
			pDC->SetPixel(upperleft.x + 9, upperleft.y + 4, RGB(0, 0, 0));
			pDC->SetPixel(upperleft.x + 9, upperleft.y + DOC_ICONHEIGHT - 4, RGB(0, 0, 0));
			
			// the hands in red
			pDC->SelectObject(&penRed);
			pDC->MoveTo(upperleft.x + 11, upperleft.y + 6);
			pDC->LineTo(upperleft.x + 7, upperleft.y + 9);
			pDC->LineTo(upperleft.x + 11, upperleft.y + 12);
			break;
	}
	
	// reset DC
	pDC->SelectObject(pOldBrush);
	pDC->SelectObject(pOldPen);
}

///////////////////////////////////////////////////////////////////////////////
// OnUpdate(): called when data change
// Arguments: lHint: Scroll to end of doc if LOWORD == TRUE, don't scroll if FALSE
//            pHint: Invalidate all if NULL. If ptr to CUvikonDoc, invalidate
//                   the rects that belong to the lines stored in the public
//                   array m_UpdateIndex
///////////////////////////////////////////////////////////////////////////////
void CListView::OnUpdate(CView* pSender, LPARAM lHint, CObject* pHint)
{
	int i, nLine;
	POINT point;
	CRect rectItem, clientRect;
	CSize docSize;
	CPoint offset;
	
	CUvikonDoc* pDoc = (CUvikonDoc*)GetDocument();
	docSize = pDoc->GetDocSize();
	SetScrollSizes(MM_TEXT, docSize);
	
	// interpret pHint
	if (pHint == NULL)
	{
		Invalidate(TRUE);
	}
	else if (pHint->IsKindOf(RUNTIME_CLASS(CUvikonDoc)))
	{
		offset = GetScrollPosition();
		for (i = 0; i < pDoc->m_UpdateIndex.GetSize(); i++)  // walk thru all lines that
		{                                                    // need to be redrawn
			rectItem.left = 0;
			nLine = pDoc->m_UpdateIndex.GetAt(i);
			rectItem.top = nLine*(m_tm.tmHeight + DOC_LINESPACE);
			rectItem.bottom = (nLine+1)*(m_tm.tmHeight + DOC_LINESPACE);
			rectItem.OffsetRect(-offset.x, -offset.y);
			GetClientRect(&clientRect);
			rectItem.right = (docSize.cx > clientRect.right) ? docSize.cx : clientRect.right;
			InvalidateRect(&rectItem, TRUE);
		}
		pDoc->m_UpdateIndex.RemoveAll();	 // clear the update index
	}
	
	// interpret lHint
	if (LOWORD(lHint))
	{
		GetClientRect(&clientRect);
		point.x = 0;
		point.y = (docSize.cy < clientRect.bottom) ? 0 : docSize.cy - clientRect.bottom;
		ScrollToPosition(point);	
	}
}

void CListView::OnInitialUpdate()
{
	CFont* pOldFont;
	
	CScrollView::OnInitialUpdate();    // call base class handler first
	
	CDC* pDC = GetDC();
	if (pDC != NULL)
	{
		pOldFont = (CFont*)pDC->SelectStockObject(/*ANSI_FIXED_FONT`*/SYSTEM_FONT);  // select font
		pDC->GetTextMetrics(&m_tm);       // get information about font
		pDC->SelectObject(pOldFont);
		CUvikonDoc* pDoc = (CUvikonDoc*)GetDocument();
		pDoc->InitializeListBox(pDC, &m_tm);
		ReleaseDC(pDC);
	}
	
	OnUpdate(NULL, 0L, NULL);
}



/////////////////////////////////////////////////////////////////////////////
// CListView message handlers

/////////////////////////////////////////////////////////////////////////////
// OnEditClear(): Handler for edit clear menu command
/////////////////////////////////////////////////////////////////////////////
void CListView::OnEditClear()
{
	CUvikonDoc* pDoc = (CUvikonDoc*)GetDocument();
	
	if (pDoc->GetNumSelected() > 0)
	{
		CDC* pDC = GetDC();
		CFont* pOldFont = (CFont*)pDC->SelectStockObject(SYSTEM_FONT);
		pDoc->EditClear(FALSE, pDC, &m_tm);
		pDC->SelectObject(pOldFont);
		ReleaseDC(pDC);
	}
}

void CListView::OnUpdateEditClear(CCmdUI* pCmdUI)
{
	pCmdUI->Enable(((CUvikonDoc*)GetDocument())->GetNumSelected() > 0);	// switch on if something is selected	
}

/////////////////////////////////////////////////////////////////////////////
// OnEditClearAll(): Handler for edit clear all menu command
/////////////////////////////////////////////////////////////////////////////
void CListView::OnEditClearAll()
{
	CUvikonDoc* pDoc = (CUvikonDoc*)GetDocument();
	
	CDC* pDC = GetDC();
	CFont* pOldFont = (CFont*)pDC->SelectStockObject(SYSTEM_FONT);
	pDoc->EditClear(TRUE, pDC, &m_tm);
	pDC->SelectObject(pOldFont);
	ReleaseDC(pDC);
}

void CListView::OnUpdateEditClearAll(CCmdUI* pCmdUI)
{
	pCmdUI->Enable(((CUvikonDoc*)GetDocument())->GetNumRun() > 0);	// switch on if there are entries	
}

/////////////////////////////////////////////////////////////////////////////
// OnEditCopy(): Handler for edit copy menu command
/////////////////////////////////////////////////////////////////////////////
void CListView::OnEditCopy()
{
	CUvikonDoc* pDoc = (CUvikonDoc*)GetDocument();
	
	if (OpenClipboard())
	{
		EmptyClipboard();	// first get rid of old clipboard data
		pDoc->CopyToClpbd();
		CloseClipboard();
	}

}

void CListView::OnUpdateEditCopy(CCmdUI* pCmdUI)
{
	pCmdUI->Enable(((CUvikonDoc*)GetDocument())->GetNumSelected() == 1);	// switch on if one item is selected	
}

/////////////////////////////////////////////////////////////////////////////
// OnEditCut(): Handler for edit cut menu command
/////////////////////////////////////////////////////////////////////////////
void CListView::OnEditCut()
{
	CUvikonDoc* pDoc = (CUvikonDoc*)GetDocument();
	
	if (OpenClipboard())
	{
		EmptyClipboard();	// first get rid of old clipboard data
		if (pDoc->CopyToClpbd())
		{
			CloseClipboard();   // close clipboard before hazardous operation
			OnEditClear();
		}
		else
		{
			CloseClipboard();
		}
	}

}

void CListView::OnUpdateEditCut(CCmdUI* pCmdUI)
{
	pCmdUI->Enable(((CUvikonDoc*)GetDocument())->GetNumSelected() == 1);	// switch on if one item is selected	
}

/////////////////////////////////////////////////////////////////////////////
// OnEditPaste(): Handler for edit paste menu command
/////////////////////////////////////////////////////////////////////////////
void CListView::OnEditPaste()
{
	CUvikonDoc* pDoc = (CUvikonDoc*)GetDocument();
	
	if (OpenClipboard())
	{
		CDC* pDC = GetDC();
		CFont* pOldFont = (CFont*)pDC->SelectStockObject(SYSTEM_FONT);
		pDoc->CopyFromClpbd(pDC, &m_tm);
		pDC->SelectObject(pOldFont);
		ReleaseDC(pDC);
		CloseClipboard();
	}
}

void CListView::OnUpdateEditPaste(CCmdUI* pCmdUI)
{
	if (OpenClipboard())
	{
		pCmdUI->Enable(IsClipboardFormatAvailable(CF_TEXT));
		CloseClipboard();	
	}
	else
	{
		pCmdUI->Enable(FALSE);	// can't access clipboard	
	}
}

/////////////////////////////////////////////////////////////////////////////
// OnEditUndo(): Handler for edit undo menu command
/////////////////////////////////////////////////////////////////////////////
void CListView::OnEditUndo()
{
	CUvikonDoc* pDoc = (CUvikonDoc*)GetDocument();
	
	pDoc->Undo();
}

void CListView::OnUpdateEditUndo(CCmdUI* pCmdUI)
{
	CUvikonDoc* pDoc = (CUvikonDoc*)GetDocument();
	
	pCmdUI->Enable(pDoc->IsUndo());	
}

/////////////////////////////////////////////////////////////////////////////
// OnKeyDown(): Handler for WM_KEYDOWN message
/////////////////////////////////////////////////////////////////////////////
void CListView::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags)
{
	CUvikonDoc* pDoc = (CUvikonDoc*)GetDocument();
	int newLine;
	
	switch(nChar)
	{
		case 38:                            // up
			newLine = pDoc->SetCurrentRunRel(FALSE);  // moves selection one line up (one index down)
			ScrollIntoVis(newLine);
			break;
		case 40:                            // down
			newLine = pDoc->SetCurrentRunRel(TRUE);   // moves selection one line down (one index up)
			ScrollIntoVis(newLine);
			break;
		case 46:                            // del
			OnEditClear();                  // clears the line
			break;
		case 113:                           // F2 key (borrowed from Excel)
			TitleEdit(pDoc->GetCurrentRun());
			break;
		default:
			// call base class handler	
			CScrollView::OnKeyDown(nChar, nRepCnt, nFlags);
	}
}

//////////////////////////////////////////////////////////////////////////////
// OnLButtonDown(): Handler for mouse left button down message
//////////////////////////////////////////////////////////////////////////////
void CListView::OnLButtonDown(UINT nFlags, CPoint point)
{
	int nLine;
	CUvikonDoc* pDoc = (CUvikonDoc*)GetDocument();
	
	// get line where mouseclick occurred  i*(m_tm.tmHeight + DOC_LINESPACE)
	point.Offset(GetScrollPosition());
	nLine = point.y/(m_tm.tmHeight + DOC_LINESPACE);
	
	if (nFlags&MK_CONTROL)       // ctrl key pressed: multiple selection of scattered entries
	{
		//m_nPrevLine = -1;
		pDoc->ResetSelExtent();
		
		pDoc->ToggleItemSelection(nLine, TRUE, FALSE);
		m_nAnchorLine = nLine;
	}
	else if (nFlags&MK_SHIFT && m_nAnchorLine != nLine)    // shift key pressed: multiple selection of a range of entries
	{
		pDoc->SelectRange(pDoc->GetCurrentRun(), nLine);       // end line of selection range
	}
	else
	{
		//m_nPrevLine = -1;
		pDoc->ResetSelExtent();
		
		pDoc->SetCurrentRun(nLine, FALSE);
		m_nAnchorLine = nLine;
	}
	
	// set capture until LButtonUp to enable multiple selection with the mouse
	SetCapture();
	
	// set prev line if mouse move follows
	m_nPrevLine = nLine;
	
}

////////////////////////////////////////////////////////////////////////
// OnLButtonUp(): Handler for mouse left button up message
////////////////////////////////////////////////////////////////////////
void CListView::OnLButtonUp(UINT nFlags, CPoint point)
{
	if (GetCapture() == this)
	{
		ReleaseCapture();
		//m_nPrevLine = -1;
		//((CUvikonDoc*)GetDocument())->ResetSelExtent();
	}
	
}

///////////////////////////////////////////////////////////////////////////
// OnMouseMove(): Handler for mouse move message
///////////////////////////////////////////////////////////////////////////
void CListView::OnMouseMove(UINT nFlags, CPoint point)
{
	int nLine;
	CUvikonDoc* pDoc = (CUvikonDoc*)GetDocument();
	
	// get line where mouse cursor currently resides  i*(m_tm.tmHeight + DOC_LINESPACE)
	point.Offset(GetScrollPosition());
	nLine = point.y/(m_tm.tmHeight + DOC_LINESPACE);
	
	if (GetCapture() == this && nLine != m_nPrevLine)   // that is, the left mouse button is pressed
	{                                                   // and the mouse has been moved from the
		pDoc->SelectRange(m_nAnchorLine, nLine);        // line of the previous call to this fn
		m_nPrevLine = nLine;
		ScrollIntoVis(nLine);
	}
	
}

///////////////////////////////////////////////////////////////////////
// OnLButtonDblClk(): Handler for mouse left button doubleclick message
///////////////////////////////////////////////////////////////////////
void CListView::OnLButtonDblClk(UINT nFlags, CPoint point)
{
	int nLine, nMaxLine;
	CUvikonDoc* pDoc = (CUvikonDoc*)GetDocument();
	CPoint offset;

	// get line where mouseclick occurred
	offset = GetScrollPosition();
	point.Offset(offset);
	nLine = point.y/(m_tm.tmHeight + DOC_LINESPACE);
	nLine = (nLine >= 0) ? nLine : 0;
	nMaxLine = pDoc->GetNumRun();
	nLine = (nLine < nMaxLine) ? nLine : nMaxLine - 1;
	
	TitleEdit(nLine);
}

////////////////////////////////////////////////////////////////////////////
// ScrollIntoVis(): Scrolls current line or line currently pointed to into
//                  the visible region of the window
////////////////////////////////////////////////////////////////////////////
void CListView::ScrollIntoVis(int nLine)
{
	CRect rect, rectItem;
	POINT pt;
	CSize maxSize;
	CUvikonDoc* pDoc = (CUvikonDoc*)GetDocument();
	maxSize = pDoc->GetDocSize();   // get size of document
	
	// get the rect of the doc which is actually visible
	CPoint ScrollPos = GetScrollPosition();
	GetClientRect(&rect);                     
	rect.OffsetRect(ScrollPos.x, ScrollPos.y);
	
	// compute the rect of the line
	rectItem.left = 0;
	rectItem.right = rect.right;        
	rectItem.top = nLine*(m_tm.tmHeight + DOC_LINESPACE);
	rectItem.bottom = (nLine+1)*(m_tm.tmHeight + DOC_LINESPACE);
	
    if (rectItem.bottom > rect.bottom)  // scroll down if necessary
    {
    	pt.x = ScrollPos.x;
    	pt.y = ScrollPos.y + (rectItem.bottom - rect.bottom);
    	pt.y = (pt.y > maxSize.cy) ? maxSize.cy : pt.y;
    	ScrollToPosition(pt);
    }
    else if (rectItem.top < rect.top)
    {                                   // scroll up if necessary
    	pt.x = ScrollPos.x;
    	pt.y = ScrollPos.y - (rect.top - rectItem.top);
    	pt.y = (pt.y > 0) ? pt.y : 0;
    	ScrollToPosition(pt);
    }
}

//////////////////////////////////////////////////////////////////////////
// TitleEdit(): Invokes dialog box for editing run title
// Argument: nLine: index of line to edit
//////////////////////////////////////////////////////////////////////////
void CListView::TitleEdit(int nLine)
{
	CTitleEditDlg dlg;
    CPoint point, offset;
	CString szTitle;
	CString* pszTitle;
	CRect rectItem, clientRect;
	CSize docSize;
	CUvikonDoc* pDoc = (CUvikonDoc*)GetDocument();
    
	// compute position of edit ctrl
	point.x = 0;
	point.y = nLine*(m_tm.tmHeight + DOC_LINESPACE);
	offset = GetScrollPosition();
	point.Offset(-offset.x, -offset.y);
	ClientToScreen(&point);
	
	// fine adjustment
	point.x += 3;
	// point.y -= 1;
	
	dlg.m_upperleft = point;
	
	// get the title string
	pszTitle = pDoc->GetRunTitle(nLine);
	if (pszTitle == NULL)
	{
		return;        // doc is empty
	}
	szTitle = *pszTitle;
	
	dlg.m_szTitle = szTitle.Right(szTitle.GetLength() - 5);   // extract title proper
	
	if (dlg.DoModal() == IDOK)    // invoke dialog box
	{
		CDC* pDC = GetDC();
		CFont* pOldFont = (CFont*)pDC->SelectStockObject(SYSTEM_FONT);
		szTitle = szTitle.Left(5) + dlg.m_szTitle;
		pDoc->SetRunTitle(nLine, &szTitle, pDC, &m_tm);   // this will cause redraw
		pDC->SelectObject(pOldFont);
		ReleaseDC(pDC);
	}
	else              // invoke redraw of line manually
	{
		rectItem.left = 0;
		rectItem.top = nLine*(m_tm.tmHeight + DOC_LINESPACE);
		rectItem.bottom = (nLine+1)*(m_tm.tmHeight + DOC_LINESPACE);
		rectItem.OffsetRect(-offset.x, -offset.y);
		docSize = pDoc->GetDocSize();
		GetClientRect(&clientRect);
		rectItem.right = (docSize.cx > clientRect.right) ? docSize.cx : clientRect.right;
		InvalidateRect(&rectItem, TRUE);
	}
}

////////////////////////////////////////////////////////////////////////
// OnRButtonDown(): Handler for mouse right button down message
//                  Used here to invoke a floating menu for edit commands
////////////////////////////////////////////////////////////////////////
void CListView::OnRButtonDown(UINT nFlags, CPoint point)
{
	CMenu menu;
	menu.CreatePopupMenu();
	if (((CUvikonDoc*)GetDocument())->IsUndo())
	{
		menu.AppendMenu(MF_ENABLED|MF_STRING, ID_EDIT_UNDO, szFMUndo);
	}
	else
	{
		menu.AppendMenu(MF_GRAYED|MF_STRING, ID_EDIT_UNDO, szFMUndo);
	}
	
	menu.AppendMenu(MF_ENABLED|MF_SEPARATOR);
	
	if (((CUvikonDoc*)GetDocument())->GetNumSelected() == 1)
	{
		menu.AppendMenu(MF_ENABLED|MF_STRING, ID_EDIT_CUT, szFMCut);
	}
	else
	{
		menu.AppendMenu(MF_GRAYED|MF_STRING, ID_EDIT_CUT, szFMCut);
	}
	
	if (((CUvikonDoc*)GetDocument())->GetNumSelected() == 1)
	{
		menu.AppendMenu(MF_ENABLED|MF_STRING, ID_EDIT_COPY, szFMCopy);
	}
	else
	{
		menu.AppendMenu(MF_GRAYED|MF_STRING, ID_EDIT_COPY, szFMCopy);
	}
	
	if (OpenClipboard())
	{
		if (IsClipboardFormatAvailable(CF_TEXT))
		{
			menu.AppendMenu(MF_ENABLED|MF_STRING, ID_EDIT_PASTE, szFMPaste);
		}
		else
		{
			menu.AppendMenu(MF_GRAYED|MF_STRING, ID_EDIT_PASTE, szFMPaste);
		}
		CloseClipboard();
	}
	else
	{
		menu.AppendMenu(MF_GRAYED|MF_STRING, ID_EDIT_PASTE, szFMPaste);
	}
	
	menu.AppendMenu(MF_ENABLED|MF_SEPARATOR);
	
	if (((CUvikonDoc*)GetDocument())->GetNumSelected() > 0)
	{
		menu.AppendMenu(MF_ENABLED|MF_STRING, ID_EDIT_CLEAR, szFMDelete);
	}
	else
	{
		menu.AppendMenu(MF_GRAYED|MF_STRING, ID_EDIT_CLEAR, szFMDelete);
	}
	
	if (((CUvikonDoc*)GetDocument())->GetNumRun() > 0)
	{
		menu.AppendMenu(MF_ENABLED|MF_STRING, ID_EDIT_CLEAR_ALL, szFMDeleteAll);
	}
	else
	{
		menu.AppendMenu(MF_GRAYED|MF_STRING, ID_EDIT_CLEAR_ALL, szFMDeleteAll);
	}
	
    ClientToScreen(&point);     // convert point from client to screen coordinates
	                            // Invoke floating pop-up menu at cursor position
	menu.TrackPopupMenu(TPM_LEFTALIGN | TPM_LEFTBUTTON, 
							point.x, point.y, this, NULL);
	CScrollView::OnRButtonDown(nFlags, point);
}

//////////////////////////////////////////////////////////////////////////////
// OnKillFocus(): Handler for WM_KILLFOCUS message
//                Invalidates view to update the selection rectangle indicating
//                the input focus
//////////////////////////////////////////////////////////////////////////////
void CListView::OnKillFocus(CWnd* pNewWnd)
{
	CPoint offset;
	CRect rectItem, clientRect;
	int nLine;
	CUvikonDoc* pDoc = (CUvikonDoc*)GetDocument();
	CSize docSize = pDoc->GetDocSize();
	
	CScrollView::OnKillFocus(pNewWnd);
	
	offset = GetScrollPosition();
	rectItem.left = 0;
	nLine = pDoc->GetCurrentRun();
	rectItem.top = nLine*(m_tm.tmHeight + DOC_LINESPACE);
	rectItem.bottom = (nLine+1)*(m_tm.tmHeight + DOC_LINESPACE);
	rectItem.OffsetRect(-offset.x, -offset.y);
	GetClientRect(&clientRect);
	rectItem.right = (docSize.cx > clientRect.right) ? docSize.cx : clientRect.right;
	InvalidateRect(&rectItem, TRUE);
}

//////////////////////////////////////////////////////////////////////////////
// OnSetFocus(): Handler for WM_SETFOCUS message
//               Invalidates view to update the selection rectangle indicating
//               the input focus
//////////////////////////////////////////////////////////////////////////////
void CListView::OnSetFocus(CWnd* pOldWnd)
{
	CPoint offset;
	CRect rectItem, clientRect;
	int nLine;
	CUvikonDoc* pDoc = (CUvikonDoc*)GetDocument();
	CSize docSize = pDoc->GetDocSize();
	
	CScrollView::OnSetFocus(pOldWnd);
	
	offset = GetScrollPosition();
	rectItem.left = 0;
	nLine = pDoc->GetCurrentRun();
	rectItem.top = nLine*(m_tm.tmHeight + DOC_LINESPACE);
	rectItem.bottom = (nLine+1)*(m_tm.tmHeight + DOC_LINESPACE);
	rectItem.OffsetRect(-offset.x, -offset.y);
	GetClientRect(&clientRect);
	rectItem.right = (docSize.cx > clientRect.right) ? docSize.cx : clientRect.right;
	InvalidateRect(&rectItem, TRUE);
}




