// tricadoc.cpp : implementation of the CTricarbDoc class
//

#include "stdafx.h"
#include "tricarb.h"

#include "tricadoc.h"
#include "mainfrm.h"
#include <afxcoll.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <ctype.h>
#include <io.h>
#include <math.h>
#include <direct.h>

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

static char iniRecover[] = "Recover";
static char iniLastFile[] = "LastFile";
static char iniSection5[] = "PowRec";

/////////////////////////////////////////////////////////////////////////////
// CTricarbDoc

IMPLEMENT_DYNCREATE(CTricarbDoc, CDocument)

BEGIN_MESSAGE_MAP(CTricarbDoc, CDocument)
	//{{AFX_MSG_MAP(CTricarbDoc)
		// NOTE - the ClassWizard will add and remove mapping macros here.
		//    DO NOT EDIT what you see in these blocks of generated code!
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CTricarbDoc construction/destruction

CTricarbDoc::CTricarbDoc()
{
	// TODO: add one-time construction code here
	m_pDataLine = NULL;          // delete m_pDataLine will not be executed on first time OnNewDoc
}

CTricarbDoc::~CTricarbDoc()
{
	int i;
	
	for (i = 0; i < m_LineArray.GetSize(); i++)
	{
		delete (CString*)m_LineArray.GetAt(i);        // delete all strings
	}
	m_LineArray.RemoveAll();                          // delete all ptrs to strings
	
	if (m_pDataLine != NULL)                          // delete data member, if existent
	{
		delete m_pDataLine;
	}
}

BOOL CTricarbDoc::OnNewDocument()
{
	int i;
	CTricarbApp* pApp;
	CString* pString;
	
	if (!CDocument::OnNewDocument())
		return FALSE;

	// (SDI documents will reuse this document)
	// reset string ptr array by deleting all CStrings the ptrs point to
	for (i = 0; i < m_LineArray.GetSize(); i++)
	{
		delete (CString*)m_LineArray.GetAt(i);
	}
    m_LineArray.RemoveAll();     // remove all ptrs, size of array is reset to zero
    
    if (m_pDataLine != NULL)     // delete old data member, if existent
    {
    	delete m_pDataLine;
    	m_pDataLine = NULL;
    }
    
    // initialize some members
    m_bIsFirstLine = TRUE;
    
	pApp = (CTricarbApp*)AfxGetApp();
	
	pString = NewLine();    // add an empty string to hold the header
	
    // if it's a recovery file, write a powerfail message to the string array
    if (pApp->GetIsRecover())
    {
    	*pString = "POWERFAIL RECOVERY\t";
    	m_bRecover = TRUE;	
    }
    else
    {
    	m_bRecover = FALSE;
    }
     
    m_sizeDoc.cx = 640;        // reset default doc size
    m_sizeDoc.cy = 480;
    m_nOutFormat = 0;
    
    // Clear panes in status line
    // we can't access status bar if it's the first document
    if (!((CTricarbApp*)AfxGetApp())->GetIsFirst())
    {
		((CMainFrame*)((CTricarbApp*)AfxGetApp())->m_pMainWnd)->UpdateStatusLine(1, "");
		((CMainFrame*)((CTricarbApp*)AfxGetApp())->m_pMainWnd)->UpdateStatusLine(2, "");
		((CMainFrame*)((CTricarbApp*)AfxGetApp())->m_pMainWnd)->UpdateStatusLine(3, "");
	}
	else
	{
		((CTricarbApp*)AfxGetApp())->FirstOff();
	}
	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// CTricarbDoc serialization

void CTricarbDoc::Serialize(CArchive& ar)
{
	if (ar.IsStoring())
	{
		// TODO: add storing code here
	}
	else
	{
		// TODO: add loading code here
	}
}

/////////////////////////////////////////////////////////////////////////////
// CTricarbDoc diagnostics

#ifdef _DEBUG
void CTricarbDoc::AssertValid() const
{
	CDocument::AssertValid();
}

void CTricarbDoc::Dump(CDumpContext& dc) const
{
	CDocument::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CTricarbDoc commands

/////////////////////////////////////////////////////////////////////////////
// NewLine(): Creates a new CString to hold a new line of data
// Return: ptr to new CString object
/////////////////////////////////////////////////////////////////////////////
CString* CTricarbDoc::NewLine()
{
	CString* pString;
	
	pString = new CString();
	m_LineArray.Add((unsigned long)pString);
	return pString;
}

/////////////////////////////////////////////////////////////////////////////
// SaveLine(): Appends new line to existing file
// Argument: pString: ptr to CString object to store
/////////////////////////////////////////////////////////////////////////////
void CTricarbDoc::SaveLine(CString* pString)
{
	char szFileName[13];
	char szprogramString[5];
	char *pszHeaderString, *pszFileString;
	int i = 0;
	CTricarbApp* pApp;
	CString string;
	CString* pHeader;
	int nTerm, nresult;
	
	
	pApp = (CTricarbApp*)AfxGetApp();	// get ptr to app
	
	if (m_bIsFirstLine)	// if it's the first entry after the header, we must create the file
	{
		m_bIsFirstLine = FALSE;      // reset first line flag
		
		if (strstr(*pString, "EOP") != NULL)           // this is the end of the running counter program
		{
			((CTricarbApp*)AfxGetApp())->SetEOPFlag();
			return;
		}
		
		if (!MakeFileName(pString))  // get a new file
		{
			pApp->WriteLog(6, "new file");	// include prog subdir into text
			// this line displays an error after a powerfail recovery, which can safely be ignored
			// in any other case, its a real error
			return;
		}
		
		// select header
		pszHeaderString = pApp->GetHeaderString(m_nOutFormat);
		nTerm = pApp->GetTerminator(m_nOutFormat);
		pHeader = (CString*)m_LineArray.GetAt(0);
		*pHeader += pszHeaderString;                // *pHeader may contain powerfail message
			
		m_pDataLine->WriteHeaderToDisk(pHeader, m_szFileName, nTerm);
		
		// save filename and set flag for powerfail recovery
		pApp->WriteProfileString(iniSection5, iniLastFile, m_szFileName);
		pApp->WriteProfileInt(iniSection5, iniRecover, (int)TRUE);

		// Update status bar
		pszFileString = strrchr(m_szFileName, (int)'\\');    // isolate filename
		((CMainFrame*)(pApp->m_pMainWnd))->UpdateStatusLine(1, pszFileString+1);
		pApp->WriteLog(3, pszFileString-2);	// include prog subdir into text
	}
	
	// isolate sample number
	strncpy(szFileName, (const char*)*pString, 12);	// szFileName recycled
	szFileName[12] = '\0';				// make a zero-terminated string
	pszHeaderString = strtok(szFileName, ",");   // pszHeaderString recycled, finds protocol#
	if (pszHeaderString != NULL)
	{
		pszHeaderString = strtok(NULL, ",");		// finds rack#
		if (pszHeaderString != NULL)
		{
			pszHeaderString = strtok(NULL, ",");		// finds sample number
			if (pszHeaderString != NULL)
			{
				_snprintf(szprogramString, 6, "S:%s", pszHeaderString);  // szProgramString recycled
				((CMainFrame*)(pApp->m_pMainWnd))->UpdateStatusLine(3, szprogramString);
			}
		}
	}
	// convert the string to our data format and save it to disk
	nresult = m_pDataLine->WriteLineToDisk(pString, m_szFileName);
	if (nresult != -1)
	{
		ReportError(nresult, NULL);
	}
	// convert the string for screen output
	ReformatScreenString(pString);         
	
	// TODO: give a hint for efficient redraw
	UpdateAllViews(NULL, (LPARAM)m_LineArray.GetSize(), NULL /*pHint*/);
}

/////////////////////////////////////////////////////////////////////////////////////
// MakeFileName(): Gets an unique filename based on date and output format
// Arguments: pstring: ptr to CString object holding a line of data
// Return: TRUE if successful, FALSE if not
/////////////////////////////////////////////////////////////////////////////////////
BOOL CTricarbDoc::MakeFileName(CString* pstring)
{
	char stringBuffer[9];
	char szFileName[16];
	char szprogramString[5];
	int i = 0;
	CTricarbApp* pApp;
	BOOL result;
	CString string;
	
	pApp = (CTricarbApp*)AfxGetApp();	// get ptr to app
	
	// we compose the filename in szFileName, the full path in m_szFileName
	// extract program no.
	strcpy(stringBuffer, pstring->Left(2));
	stringBuffer[2] = '\0';                          // make a C-type string
	if (isdigit(stringBuffer[1]))    // it's a two-digit program no
	{
		strcpy(szFileName, stringBuffer);
		m_nProgramNo = atoi(stringBuffer);  // store counter program number
	}
	else if (isdigit(stringBuffer[0]))                        // it's a one-digit program no
	{
		szFileName[0] = '0';   // leading zero
		szFileName[1] = stringBuffer[0];
		szFileName[2] = '\0';
		m_nProgramNo = (int)(stringBuffer[0] - 48); // store counter program number
	}
	else
	{
		return FALSE;
	}
	
	szprogramString[0] = 'P';
	szprogramString[1] = ':';
	strcpy(szprogramString+2, szFileName);	// keep a copy for status bar
	strcat(szFileName, "\\");               // subdir is complete now
	
	m_nOutFormat = pApp->GetOutFormat(m_nProgramNo);	// get output format
	
	switch (m_nOutFormat)                     // get appropriate data line object
	{
		case 0:
			m_pDataLine = new CPlainLine();
			break;
		case 1:
			m_pDataLine = new CExcelLine();
			break;
		case 2:
			m_pDataLine = new CMulticalcLine(1);
			break;
		case 3:
			m_pDataLine = new CMulticalcLine(2);
			break;
	}
	
	if (m_pDataLine == NULL)
	{
		return FALSE;
	}
	
	if (m_bRecover)
	{
		string = pApp->GetProfileString(iniSection5, iniLastFile, "C:\\RECOVER.TXT");
		strcpy(m_szFileName, string);	
		((CMainFrame*)(pApp->m_pMainWnd))->UpdateStatusLine(2, szprogramString);
		// TODO: perform open/write test for file. if not accessible, switch to recover.txt
		return TRUE;
	}	
	
	if ((result = m_pDataLine->MakeFileName(szFileName, m_szFileName)) == TRUE)
	{
		((CMainFrame*)(pApp->m_pMainWnd))->UpdateStatusLine(2, szprogramString);
		return TRUE;
	}
	else
	{
		return FALSE;
	}
}

/////////////////////////////////////////////////////////////////////////////////////
// ReformatScreenString(): Reformats CString to screen output format
//                   Also scans for EOP to signal end of transmission
// Arguments: pstring: ptr to CString object holding the line to reformat
/////////////////////////////////////////////////////////////////////////////////////
void CTricarbDoc::ReformatScreenString(CString* pstring)
{
	int index, nstringLength;
	char *newline, *line, *pszFileString;
	
	
	if (strstr(*pstring, "EOP") != NULL)           // this is the end of the running counter program
	{
		((CTricarbApp*)AfxGetApp())->SetEOPFlag();
		pszFileString = strrchr(m_szFileName, (int)'\\');    // isolate filename
		((CTricarbApp*)AfxGetApp())->WriteLog(4, pszFileString-2);	// include prog subdir into text
	}
	
	// Replace comma delimiters with tab delimiters
	do
	{
		index = pstring->Find(',');
		if (index != -1)                 // if the string contains a comma,...
		{
			pstring->SetAt(index, '\t'); // replace it with a tab
		}
	}
	while (index != -1);
		    
	nstringLength = pstring->GetLength();
	line = newline = pstring->GetBuffer(nstringLength+1);
		    
	// eliminate linefeeds
	do
	{
		newline = strchr(newline, (int)'\n');
		if (newline != NULL)
		{
		    memmove(newline, newline+1, strlen(newline+1)+1);   // memmove ensures data integrity when
		}	                                                  // source and destination overlap
	}
	while (newline != NULL);
		
	// eliminate trailing tab and cr
	if (strlen(line) > 2)
	{
		line[strlen(line)-2] = '\0';
	}
		    
	pstring->ReleaseBuffer();  // no argument, CString automatically resets string length
	
}

////////////////////////////////////////////////////////////////////////////////////
// GetNumLines(): Helper to access size of m_LineArray
// Returns: int number of entries in m_LineArray
////////////////////////////////////////////////////////////////////////////////////
int CTricarbDoc::GetNumLines()
{
	return m_LineArray.GetSize();		
}

////////////////////////////////////////////////////////////////////////////////////
// GetLine(): Helper to access contents of m_LineArray
// Argument: index: index of string to retrieve
// Returns: ptr to CString object at position i
////////////////////////////////////////////////////////////////////////////////////
CString* CTricarbDoc::GetLine(int index)
{
	CString* pstring;
	pstring = (CString*)(m_LineArray.GetAt(index));
	return pstring;
}

////////////////////////////////////////////////////////////////////////////////////
// SetDocSize(): Sets new document size
//               Always keeps a minimum document size to allow for some scrolling
// Argument: CSize docsize: new size of document
////////////////////////////////////////////////////////////////////////////////////
void CTricarbDoc::SetDocSize(CSize docsize)
{
	if (docsize.cx > 640)
		m_sizeDoc.cx = docsize.cx;
	if (docsize.cy > 480)
		m_sizeDoc.cy = docsize.cy;
}

////////////////////////////////////////////////////////////////////////////////////
// CanCloseFrame(): Called when document view is about to be closed.
//                  If still online, ask user if he really wants to quit
// Return: TRUE if safe to quit, FALSE if not
////////////////////////////////////////////////////////////////////////////////////
BOOL CTricarbDoc::CanCloseFrame(CFrameWnd* pFrame)
{
	if (((CTricarbApp*)AfxGetApp())->GetIsCollecting())
	{
		if (AfxMessageBox("Tricarb Monitor is still online. Do you really want to quit?", MB_YESNO) == IDYES)
		{
			return CDocument::CanCloseFrame(pFrame);
		}
		else
		{
			return FALSE;
		}
	}
	return CDocument::CanCloseFrame(pFrame);
}

/////////////////////////////////////////////////////////////////////////////////////
// ReportError(): Called by view to report COM error to log file
// Arguments: event: event ID (see CTricarbApp::WriteLog)
//            szError: ptr to string containing error message, NULL if empty
/////////////////////////////////////////////////////////////////////////////////////
void CTricarbDoc::ReportError(int event, char* szError)
{
	CTricarbApp* pApp;
	char* pszFileString;
	char szerrortext[60];
	
	pApp = (CTricarbApp*)AfxGetApp();
	pszFileString = strrchr(m_szFileName, (int)'\\');    // isolate filename
    if (event == 5)     // Transmission error
    {
		strcpy(szerrortext, pszFileString-2);  // include prog subdir into text
		strcat(szerrortext, szError);          // add error string
		pApp->WriteLog(5, szerrortext);
	}
	else if (event == 6) // output error
	{
		pApp->WriteLog(6, pszFileString-2);    // include prog subdir into text
	}
	else                // data conversion error
	{
		pApp->WriteLog(7, pszFileString-2);    // include prog subdir into text
	}	
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////
// Some support classes: CDataLine and derivatives handle data output to disk in various formats
// Hierarchy chart for support classes
//
// CDataLine
//   CTextType
//     CPlainLine
//     CExcelLine
//   CBinaryType
//     CMulticalcLine

//********************************************************************************************************
//********************************************************************************************************
// class CDataLine
CDataLine::CDataLine()
{
}

CDataLine::~CDataLine()
{
}

//////////////////////////////////////////////////////////////////////////////////
// MakeFileName(): Default implementation of virtual fn
BOOL CDataLine::MakeFileName(char* szFileName, char* szFullPath)
{
	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// WriteHeaderToDisk(): Default implementation of virtual fn
BOOL CDataLine::WriteHeaderToDisk(CString* pHeader, char* szFileName, int nTerm)
{
	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// WriteLineToDisk(): Default implementation of virtual fn
int CDataLine::WriteLineToDisk(CString* pString, char* szFileName)
{
	return -1;
}

//////////////////////////////////////////////////////////////////////////////////
// TestDirectory(): Test for the existence of the subdir
// Argument: szFileName: string containing the subdir of the data directory (e.g."02\")
// Return: TRUE if subdir exists or could be created, FALSE if subdir could not be created
//////////////////////////////////////////////////////////////////////////////////
BOOL CDataLine::TestDirectory(char* szFileName)
{
	char szfullpath[256];
	CTricarbApp* pApp;
	
	pApp = (CTricarbApp*)AfxGetApp();	// get ptr to app
	pApp->GetOutPath(szfullpath);
	
	strcat(szfullpath, szFileName);
	szfullpath[strlen(szfullpath)-1] = '\0';   // remove trailing backslash
	AnsiToOem(szfullpath, szfullpath);
	
	if (_access(szfullpath, 0))
	{
		if (_mkdir(szfullpath))
		{
			return FALSE;
		}
	}
	
	return TRUE;
}

//********************************************************************************************************
//********************************************************************************************************
// class CTextType
CTextType::CTextType()
{
}

CTextType::~CTextType()
{
}

/////////////////////////////////////////////////////////////////////////////////////
// MakeFileName: gets unique filename in Tricarb file name convention
// Arguments: szFileName: buffer for creating subdir and filename
//            szFullPath: string to hold complete path of output file
//            outFormat: output format of this
// Return: TRUE if successful, FALSE if not
///////////////////////////////////////////////////////////////////////////////////// 
BOOL CTextType::MakeFileName(char* szFileName, char* szFullPath)
{
	HFILE handle;
	OFSTRUCT openBuff;
	char stringBuffer[9];
	int i = 0;
	int index, fileexist, strlength;
	CTricarbApp* pApp;
	
	pApp = (CTricarbApp*)AfxGetApp();	// get ptr to app
	
	if (TestDirectory(szFileName))  // if subdir exists
	{
		// repeat subdir as name root
		szFileName[3] = szFileName[0];
		szFileName[4] = szFileName[1];
		
		// Get Date and convert to MMDD
		_strdate(stringBuffer);		// formatted mm/dd/yy
		szFileName[5] = stringBuffer[0];
		szFileName[6] = stringBuffer[1];
		szFileName[7] = stringBuffer[3];
		szFileName[8] = stringBuffer[4];
		szFileName[9] = '\0';
						
		// Get unique filename
		pApp->GetOutPath(szFullPath);
		strcat(szFullPath, szFileName);
		strcat(szFullPath, "00");
		strlength = strlen(szFullPath); 
		do                                        // test indices running from 00 to 99
		{
			if (i < 10)
			{
				index = strlength - 1;
			}
			else
			{
				index = strlength - 2;
			}
			_itoa(i++, &szFullPath[index], 10);
			strcpy(&szFullPath[strlength], ".TXT");
			fileexist = _access(szFullPath, 00);	// test for file existence only	
		}
		while (i < 100 && fileexist == 0);
	}
	else        // if subdir does not exist and cannot be created
	{
		strcpy(szFullPath, "C:\\RECOVER.TXT");
	}
			
	// Try to create file
	handle = OpenFile(szFullPath, &openBuff, OF_CREATE|OF_SHARE_DENY_WRITE);
			
	if (handle == HFILE_ERROR)
	{
		// TODO: Add some data safety
		return FALSE;
	}
			
	_lclose(handle);
	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////
// WriteHeaderToDisk(): Writes header to disk file
// Arguments: pHeader: ptr to CString holding header
//            szFileName: file to save to
//            nTerm: type of terminator
// Return: TRUE if successful, FALSE if not
//////////////////////////////////////////////////////////////////////////////
BOOL CTextType::WriteHeaderToDisk(CString* pHeader, char* szFileName, int nTerm)
{
	HFILE handle;
	OFSTRUCT openBuff;
	int strlength;
	BOOL berrvalue = FALSE;
	
	handle = OpenFile(szFileName, &openBuff, OF_WRITE|OF_SHARE_DENY_WRITE);
	if (handle == HFILE_ERROR)
	{
		// TODO: Add some data safety
		return FALSE;
	}
	
	// jump to end of file (append new data)
	if (_llseek(handle, 0L, 2) == -1)
	{
		// TODO: Add some data safety
		_lclose(handle);
		return FALSE;
	}

	// write header
	strlength = pHeader->GetLength();
	if (strlength > 0)     // don't write an empty header
	{
		if (_lwrite(handle, (const void huge*)(const char*)*pHeader, strlength) == HFILE_ERROR)
		{
			berrvalue = TRUE;	
		}
		if (nTerm == 0)         // add line terminator
		{
			if (_lwrite(handle, "\r\n", 2*sizeof(char)) == HFILE_ERROR)    // cr/lf
			{
				berrvalue = TRUE;
			}
		}
		else if (nTerm == 1)
		{
			if (_lwrite(handle, "\r", sizeof(char)) == HFILE_ERROR)        // cr
			{
				berrvalue = TRUE;
			}
		}
	}
	
	if (_lclose(handle) == HFILE_ERROR)
	{
		berrvalue = TRUE;
	}
	
	return !berrvalue;
}

//********************************************************************************************************
//********************************************************************************************************
// class CPlainLine
CPlainLine::CPlainLine()
{
	m_nOutFormat = 0;
}

CPlainLine::~CPlainLine()
{
}

//////////////////////////////////////////////////////////////////////////
// WriteLineToDisk(): Writes a line of data to disk
// Arguments: pString: ptr to CString holding input data line
//            szFileName: Name of file to save to
// Return: -1 if successful, 6 if not (output error)
//////////////////////////////////////////////////////////////////////////
int CPlainLine::WriteLineToDisk(CString* pString, char* szFileName)
{
	HFILE handle;
	OFSTRUCT openBuff;
	BOOL berrvalue = FALSE;


	// now we open the file and write the data
	handle = OpenFile(szFileName, &openBuff, OF_WRITE|OF_SHARE_DENY_WRITE);
	if (handle == HFILE_ERROR)
	{
		// TODO: Add some data safety
		return 6;
	}
	
	// jump to end of file (append new data)
	if (_llseek(handle, 0L, 2) == -1)
	{
		// TODO: Add some data safety
		return 6;
	}
	if (_lwrite(handle, (const void huge*)(const char*)*pString, pString->GetLength()) == HFILE_ERROR)
	{
		berrvalue = TRUE;
	}
	if (_lclose(handle) == HFILE_ERROR)
	{
		berrvalue = TRUE;
	}
    
    if (berrvalue)
    {
    	return 6;
    }
    else
    {
    	return -1;
    }
}

//********************************************************************************************************
//********************************************************************************************************
// class CExcelLine
CExcelLine::CExcelLine()
{
	m_nOutFormat = 1;
}

CExcelLine::~CExcelLine()
{
}

//////////////////////////////////////////////////////////////////////////
// WriteLineToDisk(): Writes a line of data to disk
// Arguments: pString: ptr to CString holding input data line
//            szFileName: Name of file to save to
// Return: -1 if successful, if not: 6 (output error)
//////////////////////////////////////////////////////////////////////////
int CExcelLine::WriteLineToDisk(CString* pString, char* szFileName)
{
	int index, nstringLength;
	char *newline;
	HFILE handle;
	OFSTRUCT openBuff;
	BOOL berrvalue = FALSE;

	m_szDataLine = *pString;
	// Replace comma delimiters with tab delimiters
	do
	{
		index = m_szDataLine.Find(',');
		if (index != -1)                 // if the string contains a comma,...
		{
			m_szDataLine.SetAt(index, '\t'); // replace it with a tab
		}
	}
	while (index != -1);
		    
	nstringLength = m_szDataLine.GetLength();
	newline = m_szDataLine.GetBuffer(nstringLength+1);
		    
	// eliminate linefeeds
	do
	{
		newline = strchr(newline, (int)'\n');
		if (newline != NULL)
		{
		    memmove(newline, newline+1, strlen(newline+1)+1);   // memmove ensures data integrity when
		}	                                                  // source and destination overlap
	}
	while (newline != NULL);
		    
	if(TRUE)
	{
		do
		{
			index = m_szDataLine.Find('.');
			if (index != -1)                 // if the string contains a comma,...
			{
				m_szDataLine.SetAt(index, ','); // replace it with a tab
			}
		}
		while (index != -1);
	}
		    
	m_szDataLine.ReleaseBuffer();  // no argument, CString automatically resets string length

	// now we open the file and write the data
	handle = OpenFile(szFileName, &openBuff, OF_WRITE|OF_SHARE_DENY_WRITE);
	if (handle == HFILE_ERROR)
	{
		// TODO: Add some data safety
		return 6;
	}
	
	// jump to end of file (append new data)
	if (_llseek(handle, 0L, 2) == -1)
	{
		// TODO: Add some data safety
		return 6;
	}
	if (_lwrite(handle, (const void huge*)(const char*)m_szDataLine, m_szDataLine.GetLength()) == HFILE_ERROR)
	{
		berrvalue = TRUE;
	}
	if (_lclose(handle) == HFILE_ERROR)
	{
		berrvalue = TRUE;
	}
	
	if (berrvalue)
	{
		return 6;
	}
	else
	{
		return -1;
	}
}

//********************************************************************************************************
//********************************************************************************************************
// class CBinaryType
CBinaryType::CBinaryType()
{
}

CBinaryType::~CBinaryType()
{
}
//////////////////////////////////////////////////////////////////////////////
// WriteHeaderToDisk(): Writes header to disk file
// Arguments: pHeader: ptr to CString holding header
//            szFileName: file to save to
//            nTerm: type of terminator
//////////////////////////////////////////////////////////////////////////////
BOOL CBinaryType::WriteHeaderToDisk(CString* pHeader, char* szFileName, int nTerm)
{
	return TRUE;			// in binary mode, we don't want headers
}

//********************************************************************************************************
//********************************************************************************************************
// class CMulticalcLine

CMulticalcLine::CMulticalcLine(int channels)
{
	m_OutData.id1 = 0;
	m_OutData.id2 = 0;
	m_OutData.id3 = 0;
	m_OutData.id4 = 0;
	m_OutData.id5 = 0;
	m_OutData.id6 = 0;
	if (channels == 1)
	{
		m_nOutFormat = 2;
		m_bDual = FALSE;
	}
	else
	{
		m_nOutFormat = 3;
		m_bDual = TRUE;
	}	
}

CMulticalcLine::~CMulticalcLine()
{

}

//////////////////////////////////////////////////////////////////////////
// WriteLineToDisk(): Writes a line of data to disk
// Arguments: pString: ptr to CString holding input data line
//            szFileName: Name of file to save to
// Return: -1 if successful, if not: 6 (output error); 7 (conversion error)
//////////////////////////////////////////////////////////////////////////
int CMulticalcLine::WriteLineToDisk(CString* pString, char* szFileName)
{
	char *newline, *token;
	HFILE handle;
	OFSTRUCT openBuff;
	CString szDataLine;
	BOOL berrvalue = FALSE;

	szDataLine = *pString;		// get a copy of the input string
	newline = szDataLine.GetBuffer(szDataLine.GetLength()+1);
	
	if (strstr(newline, "EOP") != NULL)		
	{
		szDataLine.ReleaseBuffer();
		return -1;
	}
	
	newline[strlen(newline)] = '\0';				// make a zero-terminated string
	token = strtok(newline, ",");   			// finds protocol#
	if (token == NULL)
	{
		szDataLine.ReleaseBuffer();
		return 7;
	}
	token = strtok(NULL, ",");					// finds rack#
	if (token == NULL)
	{
		szDataLine.ReleaseBuffer();
		return 7;
	}
	token = strtok(NULL, ",");					// finds sample number
	if (token == NULL)
	{
		szDataLine.ReleaseBuffer();
		return 7;
	}
	
	m_OutData.linenumber = (short)atoi(token);
	token = strtok(NULL, ",");					// finds counting time
	if (token == NULL)
	{
		szDataLine.ReleaseBuffer();
		return 7;
	}
	m_OutData.counttime = (short)atoi(token);
	token = strtok(NULL, ",");					// finds cpma
	if (token == NULL)
	{
		szDataLine.ReleaseBuffer();
		return 7;
	}
	m_OutData.cpma1 = (float)atof(token);
	token = strtok(NULL, ",");					// finds A:2S%
	if (token == NULL)
	{
		szDataLine.ReleaseBuffer();
		return 7;
	}
	
	if (m_bDual)
	{
		token = strtok(NULL, ",");					// finds cpmb
		if (token == NULL)
		{
			szDataLine.ReleaseBuffer();
			return 7;
		}
		m_OutData.cpmb1 = (float)atof(token);
	} 
	szDataLine.ReleaseBuffer();
	
	AutoFill();
				
	// now we open the file and write the data
	handle = OpenFile(szFileName, &openBuff, OF_WRITE|OF_SHARE_DENY_WRITE);
	if (handle == HFILE_ERROR)
	{
		// TODO: Add some data safety
		return 6;
	}
	
	// jump to end of file (append new data)
	if (_llseek(handle, 0L, 2) == -1)
	{
		// TODO: Add some data safety
		return 6;
	}
	if (_lwrite(handle, (const void huge*)(&m_OutData), sizeof(m_OutData)) == HFILE_ERROR)
	{
		berrvalue = TRUE;
	}
	if (_lclose(handle) == HFILE_ERROR)
	{
		berrvalue = TRUE;
	}
	
	if (berrvalue)
	{
		return 6;
	}
	else
	{
		return -1;
	}
}

///////////////////////////////////////////////////////////////////////////////////////
// AutoFill(): Fills the missing values in the object
//             Only id1, linenumber, cpma1, (cpmb1), and counttime must be set manually
///////////////////////////////////////////////////////////////////////////////////////
void CMulticalcLine::AutoFill()
{
	// TODO: set counting time to -60??
	m_OutData.cpma2 = m_OutData.cpma1;
	if (m_OutData.cpma2 > 0)
	{
		m_OutData.sqrtcpma2 = (float)sqrt((double)m_OutData.cpma2);
	}
	else
	{
		m_OutData.sqrtcpma2 = 0;
	}
	
	if (m_bDual)
	{
		m_OutData.cpmb2 = m_OutData.cpmb1;
		if (m_OutData.cpmb2 > 0)
		{
			m_OutData.sqrtcpmb2 = (float)sqrt((double)m_OutData.cpmb2);
		}
		else
		{
			m_OutData.sqrtcpmb2 = 0;
		}
	}
	else
	{
		*((long*)&m_OutData.cpmb1) = -16777215L;  // a slimy hack to get a long in place of a float
		m_OutData.cpmb2 = m_OutData.cpmb1;
		m_OutData.sqrtcpmb2 = m_OutData.cpmb2;
	}
		
}

/////////////////////////////////////////////////////////////////////////////////////
// MakeFileName: gets unique filename in Multicalc input file name convention
// Arguments: szFileName: buffer for creating subdir and filename
//            szFullPath: string to hold complete path of output file
//            outFormat: output format of this
// Return: TRUE if successful, FALSE if not
///////////////////////////////////////////////////////////////////////////////////// 
BOOL CMulticalcLine::MakeFileName(char* szFileName, char* szFullPath)
{
	HFILE handle;
	OFSTRUCT openBuff;
	char stringBuffer[9];
	char dd[3];
	char mm[3];
	char yy[5];
	char month0, month1;
	int i = 0;
	int index, fileexist, strlength, nyear, nmonths, ndays;
	CTricarbApp* pApp;
	
	pApp = (CTricarbApp*)AfxGetApp();	// get ptr to app
	
	if (TestDirectory(szFileName))    // if subdir exists
	{
		// Get Date and convert to MultiCalc
		_strdate(stringBuffer);		// formatted mm/dd/yy
		mm[0] = stringBuffer[0];
		mm[1] = stringBuffer[1];
		mm[2] = '\0';
		dd[0] = stringBuffer[3];
		dd[1] = stringBuffer[4];
		dd[2] = '\0';
		yy[2] = stringBuffer[6];
		yy[3] = stringBuffer[7];
		yy[4] = '\0';
				
		nyear = atoi(&yy[2]);
		if (nyear < 80)
		{
			yy[0] = 50;
			yy[1] = 48;
		}
		else
		{
			yy[0] = 49;
			yy[1] = 57;
		}
		nyear = atoi(yy);
		nmonths = atoi(mm);
		nmonths += (nyear-1980)*12;
				
		month0 = (char)(nmonths/36);
		month1 = (char)(nmonths%36);
		month0 += (month0 < 10) ? 48 : 55;
		month1 += (month1 < 10) ? 48 : 55;
				
		ndays = atoi(dd);
		ndays += (ndays < 10) ? 48 : 55;
				
		szFileName[3] = month0;
		szFileName[4] = month1;
		szFileName[5] = (char)ndays;
		szFileName[6] = '\0';
				
		strcat(szFileName, "0100");       // 01 = version, 00 = run id
						
		// Get unique filename
		pApp->GetOutPath(szFullPath);
		strcat(szFullPath, szFileName);
		strlength = strlen(szFullPath); 
		do                                        // test indices running from 00 to 99
		{
			if (i < 10)
			{
				index = strlength - 1;
			}
			else
			{
				index = strlength - 2;
			}
			_itoa(i++, &szFullPath[index], 10);
			strcpy(&szFullPath[strlength], "1.DA");      // 1 = manual input file
			fileexist = _access(szFullPath, 00);	// test for file existence only	
		}
		while (i < 100 && fileexist == 0);
	}
	else   // if subdir cannot be found and cannot be created
	{
		strcpy(szFileName, "C:\\RECOVER.TXT");
	}
			
	// Try to create file
	handle = OpenFile(szFullPath, &openBuff, OF_CREATE|OF_SHARE_DENY_WRITE);
			
	if (handle == HFILE_ERROR)
	{
		// TODO: Add some data safety
		return FALSE;
	}
			
	_lclose(handle);
	return TRUE;
}

	


